///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#include "log.h"
#include "memstat.h"

#include <hash_map.h> // STL
#include <iostream.h>

// Return ceil(log_2(num)).
// num must be positive.
static int lg (int num)
{
  assert (num > 0);
  int power = 0;
  int n = 1;
  // Invariant: 2^power == n.
  while (n < num) {
    n <<= 1;
    power++;
  }
  return power;
}


struct eqUL
{
  bool operator()(unsigned long s1, unsigned long s2) const
  {
    return s1 == s2;
  }
};


int main (int argc,
	  char *argv[])
{
  const int SIZE_CLASSES = 28;

  int inUse = 0;
  int maxInUse = 0;
  int allocated = 0;
  int maxAllocated = 0;
  int inUseAtMaxAllocated = 0;

  int numberOfObjects = 0;
  int totalObjectSize = 0;

  Log<MemoryRequest> l, m;
  hash_map<unsigned long, int, hash<unsigned long>, eqUL> table;

  int histogram[SIZE_CLASSES];
  int inUseHistogram[SIZE_CLASSES];

  for (int i = 0; i < SIZE_CLASSES; i++) {
    histogram[i] = 0;
    inUseHistogram[i] = 0;
  }

  int NLOGS = 128;
  int MULTIPLIER = 1;

  char l_filename[255];

  int log = NLOGS;

  if (argc > 1) {
    strncpy (l_filename, argv[1], 255);
  } else {
    sprintf (l_filename, "log%d", log);
  }

  l.open (l_filename, Log<MemoryRequest>::READ_ONLY);
  int i = 0;

  double prevTime = 0.0;

  while (i < l.length()) {

    printf (".");

    double currTime = l[i].getTime();
    assert (prevTime <= currTime);
    prevTime = currTime;
    int sz = l[i].getSize();
    int lgSz;
    unsigned long addr = l[i].getAddress();

    switch (l[i].getType()) {

    case MemoryRequest::MALLOC_OP:

      if (sz == 0) {
	sz = sizeof(double);
      }
      inUse += sz * MULTIPLIER;
      assert (inUse <= allocated);
      if (inUse > maxInUse) {
	maxInUse = inUse;
      }
      table[addr] = sz * MULTIPLIER;
      numberOfObjects++;
      totalObjectSize += sz;
      lgSz = lg(sz);
      histogram[lgSz]++;
      inUseHistogram[lgSz]++;
      break;

    case MemoryRequest::FREE_OP:

      if (table[addr] != 0) {
	inUse -= table[addr] * MULTIPLIER;
	if (lg(table[addr] / MULTIPLIER) != 0) {
	  inUseHistogram[lg(table[addr] / MULTIPLIER)]--;
	}
      } else {
	assert(0);
      }
      break;

    case MemoryRequest::ALLOCATE_OP:
      
      allocated += l[i].getAllocated() * MULTIPLIER;
      if (allocated > maxAllocated) {
	maxAllocated = allocated;
	inUseAtMaxAllocated = inUse;
      }
      break;

    case MemoryRequest::DEALLOCATE_OP:

      allocated -= l[i].getDeallocated() * MULTIPLIER;
      break;

    default:
      // We should never get here.
      printf ("yow!\n");
      assert (0);
      exit (1);
      // cout << "Doh!" << endl;
      break;
    }
    // cout << "(in use = " << inUse << ")" << endl;
    i++;
  }

  l.close();

#if 0
  for (int i = 0; i < SIZE_CLASSES; i++) {
    cout << (1 << i) << ": " << histogram[i] << " (still allocated = " << inUseHistogram[i] << ") " << endl;
  }
#endif

  totalObjectSize = totalObjectSize / MULTIPLIER;
  allocated = allocated / MULTIPLIER;
  inUseAtMaxAllocated = inUseAtMaxAllocated / MULTIPLIER;
  maxAllocated = maxAllocated / MULTIPLIER;
  inUse = inUse / MULTIPLIER;
  maxInUse = maxInUse / MULTIPLIER;
  cout << "Total number of objects = " << numberOfObjects << endl;
  cout << "Total object size = " << totalObjectSize << endl;
  cout << "Average object size = " << (double) totalObjectSize / (double) numberOfObjects << endl;
  cout << "Allocated = " << allocated << endl;
  cout << "Max allocated = " << maxAllocated << endl;
  cout << "Max in use = " << maxInUse << endl;
  cout << "In use at max allocated = " << inUseAtMaxAllocated << endl;
  cout.precision(10);
  cout << "Fragmentation (measure 3) = " << (double) maxAllocated / (double) inUseAtMaxAllocated << endl;
  cout << "'Real' fragmentation (measure 4) = " << (double) maxAllocated / (double) maxInUse << endl;
  cout << "Still in use = " << inUse << endl;
}
