///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

#include "log.h"
#include "log2.h"
#include "memstat.h"

#include <assert.h>
#include <iostream.h>
#include <unistd.h>
#include <stdlib.h>


void mergeLogs (Log2<MemoryRequest>& l1,
		Log2<MemoryRequest>& l2,
		Log<MemoryRequest>& outLog)
{
  int i = 0;
  int j = 0;

  double currTime = 0.0;
  double prevTime = 0.0;

#if 0
  // Sanity checks!
  for (i = 0; i < l1.length(); i++) {
    prevTime = currTime;
    currTime = l1[i].getTime();
    assert (prevTime <= currTime);
  }
  currTime = 0.0;
  for (i = 0; i < l2.length(); i++) {
    prevTime = currTime;
    currTime = l2[i].getTime();
    assert (prevTime <= currTime);
  }
#endif
  
  i = 0;
  j = 0;
  currTime = 0.0;
  prevTime = 0.0;
  while ((i < l1.length()) && (j < l2.length())) {
    double l1time = l1[i].getTime();
    double l2time = l2[j].getTime();
    prevTime = currTime;
    if (l1time < l2time) {
      currTime = l1time;
      assert (prevTime <= currTime);
      outLog.append (l1[i]);
      i++;
    } else if (l1time == l2time) {
      // Given the choice between two operations,
      // put allocations first. This is conservative.
      if ((l1[i].getType() == MemoryRequest::MALLOC_OP)
	  || (l1[i].getType() == MemoryRequest::REALLOC_OP)
	  || (l1[i].getType() == MemoryRequest::ALLOCATE_OP)) {
	currTime = l1time;
	assert (prevTime <= currTime);
	outLog.append (l1[i]);
	i++;
      } else {
	currTime = l2time;
	assert (prevTime <= currTime);
	outLog.append (l2[j]);
	j++;
      }

    } else {
      currTime = l2time;
      assert (prevTime <= currTime);
      outLog.append (l2[j]);
      j++;
    }
  }

  if (j >= l2.length()) {
    while (i < l1.length()) {
      prevTime = currTime;
      currTime = l1[i].getTime();
      assert (prevTime <= currTime);
      outLog.append (l1[i]);
      i++;
    }
  } else {
    while (j < l2.length()) {
      prevTime = currTime;
      currTime = l2[j].getTime();
      assert (prevTime <= currTime);
      outLog.append (l2[j]);
      j++;
    }
  }
  assert (outLog.length() == l1.length() + l2.length());
}


int main (int argc, char *argv[])
{
  Log2<MemoryRequest> l, m;
  Log<MemoryRequest> tmp;

  char tmpfname[255];
  sprintf (tmpfname, "memory-log%d", getpid());

  int NLOGS = 128;

  int start = 0;
  int stop = NLOGS;

  if (argc > 2) {
    start = atoi(argv[1]);
    stop = atoi(argv[2]);
  }

  printf ("Running from %d to %d\n", start, stop);

  for (int i = start; i < stop; i++) {
    printf (".");
    fflush (stdout);

    unlink (tmpfname);
    tmp.open (tmpfname);
    char l_filename[255];
    sprintf (l_filename, "log%d", i);
    char m_filename[255];
    sprintf (m_filename, "log%d", i + 1);
    // printf ("merging %s and %s\n", l_filename, m_filename);
    m.open (m_filename, Log<MemoryRequest>::READ_ONLY);
    if (m.length() == 0) {
      m.abort();
      unlink (m_filename);
      rename (l_filename, m_filename);
    } else {
      l.open (l_filename, Log<MemoryRequest>::READ_ONLY);
      if (l.length() == 0) {
	l.abort();
	m.abort();
	// Do nothing.
	// m already contains the merged contents of both files.
      } else {
	mergeLogs (l, m, tmp);
	l.abort();
	m.abort();
	tmp.close();
	unlink (m_filename);
	rename (tmpfname, m_filename);
      }
    }
    unlink (l_filename);
  }
  printf ("Done.\n");
}
